<?php

namespace App\Http\Controllers\Edu\Sms;

use App\Models\Edu\NonSite\AllInstituteList;
use App\Models\Edu\SMS\EduSms;
use Auth;
use Carbon\Carbon;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Http\JsonResponse;
use App\Traits\RequestHandlingWithMachine;
use App\Models\Edu\Device\DeviceLog;
use App\Models\User\EmployeeHistory;
use App\Models\User\GlobalEmployeeHistory;
use Illuminate\Validation\ValidationException;
use Error;
use Exception;
use Response;
use Session;
use App\Traits\SmsFunctionsTrait;

class EduSmsController extends Controller
{

    use  SmsFunctionsTrait;
    private $sms;

    public function __construct(EduSms $sms)
    {
        $this->middleware('auth');
        $this->middleware('eduworlduserchecker');
        $this->sms = $sms;
    }


    //    public function index()
    //    {
    //        $viewType = 'Sms Master View';
    //        return view('default.admin.layouts.master', compact('viewType'));
    //    }


    public function groupGetEmployeeStudents(Request $request)
    {
        try {
            $paginate = $request->paginate ?? 20;
            $search_txt = $request->search_txt;
            $txt = '%' . $search_txt . '%';
            if ($request->user_type == 1) {
                $paginator = GlobalEmployeeHistory::where('site_id', $request->site_id)
                    ->when($search_txt, function ($q) use ($txt) {
                        return $q->where(function ($query) use ($txt) {
                            $query->where('username', 'LIKE', $txt)
                                ->orWhere('email', 'LIKE', $txt)
                                ->orWhere('contact_number', 'LIKE', $txt)
                                ->orWhereTranslationLike('first_name', $txt)
                                ->orWhereTranslationLike('last_name', $txt);
                        });
                    })
                    ->where('status', true)
                    ->with('employee')
                    ->paginate($paginate);

                $transformed = $paginator->getCollection()->map(function ($item) {
                    return [
                        'name' => $item->full_name,
                        'mobile_no' => $item->employee->contact_number ?? null,
                        'email' => $item->employee->email ?? null,
                    ];
                });


                // Replace the original collection with the transformed one
                $paginator->setCollection($transformed);

                return $paginator;
            } else {
                return 'student';
            }
        } catch (ValidationException $exception) {
            return JsonResponse::create(['message' => $exception->getMessage(), 'errors' => $exception->validator->getMessageBag()->toArray()], 422);
        } catch (Exception $exception) {
            throw new Error($exception->getMessage());
        }
    }

    public function createSms(Request $request)
    {

        try {

            $this->validate($request, [
                'sms_title' => 'required',
                'sms_description' => 'required',
            ]);

            $this->sms = new EduSms();

            foreach ($this->sms->ownFields as $k => $ownField) {
                if ($request->{$ownField}) {
                    $this->sms->{$ownField} = $request->{$ownField};
                }
            }

            $this->sms->status = $request->status;
            $this->sms->created_by = Auth::id();
            $this->sms->save();
            return response()->json(['data' => $this->sms, 'message' => $this->sms->sms_title . ' Successfully Saved!', 'mode' => 'Save'], 200);
        } catch (ValidationException $exception) {
            return JsonResponse::create(['message' => $exception->getMessage(), 'errors' => $exception->validator->getMessageBag()->toArray()], 422);
        } catch (Exception $exception) {
            throw new Error($exception->getMessage());
        }
    }



    public function update(Request $request)
    {

        try {

            $this->validate($request, [
                'sms_title' => 'required',
                'sms_description' => 'required',
            ]);

            $this->sms = $this->sms->findOrFail($request->id);


            foreach ($this->sms->ownFields as $k => $ownField) {
                if ($request->{$ownField}) {
                    $this->sms->{$ownField} = $request->{$ownField};
                }
            }

            $this->sms->status = $request->status;
            $this->sms->save();
            return response()->json(['data' => $this->sms, 'message' => $this->sms->sms_title . ' Successfully Updated!', 'mode' => 'Update'], 200);
        } catch (ValidationException $exception) {
            return JsonResponse::create(['message' => $exception->getMessage(), 'errors' => $exception->validator->getMessageBag()->toArray()], 422);
        } catch (Exception $exception) {
            throw new Error($exception->getMessage());
        }
    }




    public function getListWithPagination(Request $request)
    {
        try {
            $status = $request->status;
            $search_txt = $request->search_txt;
            $date_range = $request->date_range ? json_decode($request->date_range) : null;
            $paginate = $request->paginate ?? 20;
            $txt = '%' . $search_txt . '%';

            if (!$date_range) {
                throw new Error('Please select Date');
            }


            $date_start = $date_range->start; //.' 00:00:00';
            $date_end = $date_range->end ?? $date_start;

            $list = $this->sms
                ->where('status', $status)
                ->whereDate('created_at', '>=', $date_start)
                ->whereDate('created_at', '<=', $date_end)
                ->when($search_txt, function ($q) use ($txt) {
                    return $q->where(function ($query) use ($txt) {
                        $query->where('doc_number', 'LIKE', $txt)
                            ->orWhere('sms_title', 'LIKE', $txt);
                    });
                })
                ->orderBy('id', 'DESC')
                ->paginate($paginate);

            return response()->json($list, 200);
        } catch (Exception $exception) {
            throw new Error($exception->getMessage());
        }
    }



    public function delete($id)
    {
        try {

            $the_sms = $this->sms->findOrFail($id);
            $the_sms->delete();
            return response()->json(['message' => $the_sms->sms_title . ' Successfully Deleted'], 200);
        } catch (ModelNotFoundException $ex) {
            return Response::json(['message' => 'The Notice Not Found!'], 405);
        } catch (Exception $ex) {
            return Response::json(['message' => 'Something went Wrong!'], 405);
        }
    }





    //Send SMS to Institute

    public function sendToInstitute(Request $request)
    {

        try {



            $data = [];
            $data['academic_group_id'] = $request->academic_group_id;
            $data['division_id'] = $request->division_id;
            $data['district_id'] = $request->district_id;
            $data['thana_id'] = $request->thana_id;
            $data['academic_education_level_id'] = $request->academic_education_level_id;
            $data['academic_institute_management_type_id'] = $request->academic_institute_management_type_id;
            $data['gender_id'] = $request->gender_id;
            $data['accepted_as_site'] = $request->accepted_as_site;

            $sms_id = $request->sms_id;
            $confirmed = $request->confirmed;

            if (count($request->all_institute_id) <= 0) {
                throw new Error('Please select Institute !');
            }


            if (count($request->all_institute_id) > 0) {
                $number_array = AllInstituteList::whereIn('id', $request->all_institute_id)
                    ->pluck('institution_mobile')->filter()
                    ->toArray();
            } else {
                $number_array = AllInstituteList::when($data['academic_group_id'], function ($query) use ($data) {
                    return $query->where('academic_group_id', $data['academic_group_id']);
                })
                    ->when($data['division_id'], function ($query) use ($data) {
                        return $query->where('division_id', $data['division_id']);
                    })
                    ->when($data['district_id'], function ($query) use ($data) {
                        return $query->where('district_id', $data['district_id']);
                    })
                    ->when($data['thana_id'], function ($query) use ($data) {
                        return $query->where('thana_id', $data['thana_id']);
                    })
                    ->when(!empty($data['academic_education_level_id']), function ($query) use ($data) {
                        return $query->where('academic_education_level_id', $data['academic_education_level_id']);
                    })
                    ->when($data['academic_institute_management_type_id'], function ($query) use ($data) {
                        return $query->where('academic_institute_management_type_id', $data['academic_institute_management_type_id']);
                    })
                    ->when($data['gender_id'], function ($query) use ($data) {
                        return $query->where('gender_id', $data['gender_id']);
                    })
                    ->where('accepted_as_site', $data['accepted_as_site'])
                    ->where('status', true)
                    ->pluck('institution_mobile')
                    ->filter()
                    ->toArray();
            }




            if (!$request->sms_company) {
                throw new Error('Please select SMS Company !');
            }

            if (count($number_array) < 1) {
                throw new Error('No Institute found for the condition !');
            }

            //            return $number_array;

            $sms_masking = $request->sms_masking;
            $sms_company = $request->sms_company;

            if (!$confirmed) {
                $the_sms = $this->sms->findOrFail($sms_id);
                $sms_count = $this->checkSmsCount($the_sms->sms_description);
                $sms_balance_needed = ($sms_count * ($sms_masking ? 0.40 : 0.15) * count($number_array));

                return Response::json(['message' => 'You are going to send:' . count($number_array) . ' Institute & sms count:' . $sms_count . ' & balance needed:' . $sms_balance_needed], 412);
            } else {


                $the_sms = $this->sms->findOrFail($sms_id);
                $total_sms_cost_details = $this->sendEduSmsUsingQueue(
                    $number_array,
                    $the_sms->sms_description,
                    $sms_company,
                    $sms_masking
                );

                return Response::json(['mode' => 'Success', 'message' => $total_sms_cost_details], 200);
                //                return Response::json(['mode' => 'Success', 'message' => ''], 200);
            }
        } catch (ModelNotFoundException $exception) {
            throw new Error($exception->getMessage());
        } catch (Exception $exception) {
            throw new Error($exception->getMessage());
        }
    }



    public function checkSmsCount($txt) //type means english or unicode
    {

        if (!$txt) {
            // throw new Error('Sms Text Or Type Not Found!');
            return 0;
        }
        $message_text = str_replace("[nl]", "  ", $txt);

        if (strlen($txt) != strlen(utf8_decode($txt))) //if unicode chang
        {
            $length = mb_strlen($message_text);
            $count = ceil($length / 70);
        } else { //english sms

            $length = mb_strlen($message_text);
            $count = ceil($length / 160);
        }

        // dd($txt,$message_text,$count);
        return $count;
    }


    private function logSmsSentDetails($the_sms, $sent_log)
    {
        $time = Carbon::now()->toDateTime()->format('d M Y g:i A');

        $the_sms->sent_request = ($the_sms->sent_request . ' <br/>' . $sent_log . ' <i class="fa fa-clock-o"></i>' . $time);
        $the_sms->save();

        return $the_sms;
    }
}
