<?php

namespace App\Http\Controllers\Site\Routine;

use App\Forms\DataHelper;
use App\Http\Controllers\ProjectController;
use App\Jobs\SendSmsJob;
use App\Models\Employee\SiteEmployeeDepartmentClassSubject;
use App\Models\Site\Academic\SitePeriodTypeDuration;
use App\Models\Site\Employee\Employee;
use App\Models\Site\Routine\RoutineAllocation;
use App\Models\Site\Routine\RoutineDetail;
use App\Models\Site\Routine\RoutineSwapCancel;
use App\Traits\Site\Routine\RoutineFunction;
use Carbon\Carbon;
use Error;
use Exception;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Validation\ValidationException;
use Response;

class SwapCancelController extends ProjectController
{

    use  RoutineFunction;

    private $routineDetail;
    private $routineAllocation;
    private $swapCancel;
    private $employee;


    public function __construct(RoutineDetail $routineDetail, RoutineAllocation $routineAllocation, RoutineSwapCancel $swapCancel, Employee $employee)
    {

        $this->middleware('auth');
        $this->middleware('sitepagechecker');
        $this->routineDetail = $routineDetail;
        $this->routineAllocation = $routineAllocation;
        $this->swapCancel = $swapCancel;
        $this->employee = $employee;


    }

    public function swapClass(Request $request)
    {
        try {


            $this->validate($request, [
                'swap_cancel_date' => 'required',
                'subject_id' => 'required|not_in:0',
                'teacher_id' => 'required|not_in:0',
            ]);
            $allocation_id = $request->id;
            $swap_cancel_date = $request->swap_cancel_date;
            $subject_id = $request->subject_id;
            $teacher_id = $request->teacher_id;
            $room_id = $request->room_id;
            $send_sms = $request->sendSms;

            //check if the allocation is already cancelled or swapped

            $the_swap_cancel = $this->swapCancel
                ->where('allocation_id', $allocation_id)
                ->first();

            if ($the_swap_cancel) {
                throw new Error('Already Swapped Or Cancelled!');
            }


            $the_allocation = $this->routineAllocation
                ->where('id', $allocation_id)
                ->first();


            $this->swapCancel->allocation_id = $allocation_id;
            $this->swapCancel->swap_cancel_date = $swap_cancel_date;
            $this->swapCancel->subject_id = $subject_id ?? $the_allocation->subject_id ?? 0;
            $this->swapCancel->employee_id = $teacher_id;
            $this->swapCancel->room_id = $room_id ?? $the_allocation->room_id ?? 0;
            $this->swapCancel->is_cancelled = 0;
            $this->swapCancel->swap_cancel_reason = "";
            $this->swapCancel->save();


            if ($send_sms) {

                $the_emp = $this->employee->findOrFail($teacher_id);
                if ($the_emp->contact_number) {

                    $the_site = $the_emp->siteInfo;
                    $data = "Class Swapped For You";
                    $data .= 'nl' . 'Regards: ' . $the_site->site_name;

                    $att_data_for_queue = [];
                    $att_data_for_queue['ins_type'] = 'site';
                    $att_data_for_queue['ins_id'] = $the_site->id;

                    dispatch(new SendSmsJob([$the_emp->contact_number], $data, $att_data_for_queue));

                    return response()->json(['message' => 'Successfully Swapped By Sending SMS', 'mode' => 'Success'], 200);

                }
            }


            return response()->json(['message' => 'Successfully Swapped', 'mode' => 'Success'], 200);


        } catch (ValidationException $exception) {
            return JsonResponse::create(['message' => $exception->getMessage(), 'errors' => $exception->validator->getMessageBag()->toArray()], 422);
        } catch (Exception $exception) {
            return JsonResponse::create(['message' => $exception->getMessage()], 410);

        }

    }


    public function cancelClass(Request $request)
    {
        try {


            //   \Log::info($request);

            $this->validate($request, [
                'swap_cancel_date' => 'required',
                'id' => 'required|not_in:0',
            ]);
            $allocation_id = $request->id;
            $swap_cancel_date = $request->swap_cancel_date;

            //check if the allocation is already cancelled or swapped

            $the_swap_cancel = $this->swapCancel
                ->where('allocation_id', $allocation_id)
                ->where('is_cancelled', true)
                ->first();

            if ($the_swap_cancel) {
                throw new Error('Already Cancelled!');
            }


            $the_swap_cancel = $this->swapCancel
                ->where('allocation_id', $allocation_id)
                ->where('swap_cancel_date', $swap_cancel_date)
                ->first();

            if ($the_swap_cancel) {
                $this->swapCancel = $the_swap_cancel;
            }

            $this->swapCancel->allocation_id = $allocation_id;
            $this->swapCancel->swap_cancel_date = $swap_cancel_date;
            $this->swapCancel->is_cancelled = true;
            $this->swapCancel->swap_cancel_reason = "No Reason";
            $this->swapCancel->save();


            return response()->json(['message' => 'Successfully Cancelled', 'mode' => 'Success'], 200);


        } catch (ValidationException $exception) {
            return JsonResponse::create(['message' => $exception->getMessage(), 'errors' => $exception->validator->getMessageBag()->toArray()], 422);
        } catch (Exception $exception) {
            return JsonResponse::create(['message' => $exception->getMessage()], 410);

        }

    }

    public function revertClass(Request $request)
    {
        try {


            $this->validate($request, [
                'swap_cancel_date' => 'required',
                'id' => 'required|not_in:0',
            ]);
            $swap_cancel_id = $request->id;
            $swap_cancel_date = $request->swap_cancel_date;

            //check if the revert is valid
            $now = Carbon::now()->toDateString();
            if ($now > $swap_cancel_date) {
                throw new Error('Not Allowed To Revert Previous Date Class!');
            }

            $the_swap_cancel = $this->swapCancel
                ->where('id', $swap_cancel_id)
                ->first();


            $the_swap_cancel->delete();


            return response()->json(['message' => 'The Class Successfully Reverted', 'mode' => 'success'], 200);


        } catch (ValidationException $exception) {
            return JsonResponse::create(['message' => $exception->getMessage(), 'errors' => $exception->validator->getMessageBag()->toArray()], 422);
        } catch (Exception $exception) {
            return JsonResponse::create(['message' => $exception->getMessage()], 410);

        }

    }


    public function getTeacherToSwap(Request $request)
    {

        try {


            $allocation_id = $request->allocation_id;
            $emp_type = $request->emp_type;//1-all 2-available
            $slot_date = $request->slot_date;
            $subject_id = $request->subject_id;

            $the_allocation = $this->routineAllocation->find($allocation_id);

            $academic_year_id = $the_allocation->routineDetail->academic_year_id;
            $academic_department_id = $the_allocation->routineDetail->academic_department_id;
            $academic_class_id = $the_allocation->routineDetail->academic_class_id;

            $day = date('l', strtotime($slot_date));//sun

            $get_period_type_id = (new SitePeriodTypeDuration())
                    ->where('start_date', '<=', $slot_date)
                    ->where('end_date', '>=', $slot_date)
                    ->first()->academic_period_type_id ?? 0;

            if (!$get_period_type_id) {
                throw new Error('No Period Type Found For: ' . $day . ' On ' . $slot_date);
            }


            $emp_ids = SiteEmployeeDepartmentClassSubject::whereNull('release_date')
                ->when($academic_department_id, function ($query) use ($academic_department_id) {
                    return $query->where('academic_department_id', $academic_department_id);
                })
                ->where('academic_class_id', $academic_class_id)
                ->where('academic_subject_id', $subject_id)
                ->where('user_id', '!=', $the_allocation->employee_id)
                ->pluck('user_id');


            $available_emp_ids = [];
            if ($emp_type == 1) {//all
                $available_emp_ids = $emp_ids;

            } elseif ($emp_type == 2) {//available

                foreach ($emp_ids as $k => $emp_id) {
                    $available_class = $this->getEmpAvailableClass($academic_year_id, $get_period_type_id, $emp_id);
                    if ($available_class) {
                        $available_emp_ids[] = $emp_id;
                    }
                }
            }

            $emp_list = Employee::whereIn('id', $available_emp_ids)
                ->select(['id', 'employee_id'])
                ->get();
            if (count($emp_list) < 1) {
                throw new Error('No Teacher Found For The Subject');

            }

            return response()->json([$emp_list], 200);


        } catch (Exception $exception) {

            throw new Error($exception->getMessage());

        }


    }


    public function getClassWiseRoutineByDate(Request $request)
    {

        $message = 'Nothing';
        $data = [];
        $data['academic_version_id'] = $request->academic_version_id == 'null' ? null : $request->academic_version_id;
        $data['academic_year_id'] = $request->academic_year_id == 'null' ? null : $request->academic_year_id;
        $data['academic_shift_id'] = $request->academic_shift_id == 'null' ? null : $request->academic_shift_id;
        $data['academic_department_id'] = $request->academic_department_id == 'null' ? null : $request->academic_department_id;
        $data['academic_class_id'] = $request->academic_class_id == 'null' ? null : $request->academic_class_id;
        $data['academic_section_id'] = $request->academic_section_id == 'null' ? null : $request->academic_section_id;
        $data['academic_class_group_id'] = $request->academic_class_group_id == 'null' ? null : $request->academic_class_group_id;
        $data['academic_session_id'] = $request->academic_session_id == 'null' ? null : $request->academic_session_id;
        $data['academic_period_type_id'] = $request->academic_period_type_id == 'null' ? null : $request->academic_period_type_id;

        $date = $request->swap_cancel_date;

        if (!$date) {
            throw new Error('Please Select Date');
        }

        $this->checkClassSection($data);
        //=======================================-------------==

        $slot_data = $this->routineSlotsByDateForSwapCancel($data, $date);

        $day = date('l', strtotime($date));


        $class_section_name = 'Dept:' . ($slot_data[0]->dept->name ?? 'n/a') . ' Class:' . ($slot_data[0]->academicClass->class_name ?? 'n/a') .
            ' Group:' . ($slot_data[0]->classGroup->group_name ?? 'n/a') . ' Section:' . ($slot_data[0]->academicSection->section_name ?? '');


        return response()->json([$slot_data, $date, $day, $class_section_name], 200);

        //  dd($slot_data);


    }


}
