<?php

namespace App\Http\Controllers\Site;


use App\Http\Controllers\ProjectController;
use App\Mail\SendMailToUserWithQueue;
use App\Models\Site\SiteInfo;
use App\Models\Site\Student\Student;
use App\Models\User\User;
use App\Traits\Site\UserPermission\UserAccessTrait;
use Auth;
use DB;
use Error;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use App\Http\Controllers\Language\Lang;
use Illuminate\Validation\ValidationException;
use Mail;
use Response;
use Exception;
use Session;

class SiteUserController extends ProjectController
{


    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     *
     *
     */
    use UserAccessTrait, Lang;

    private $user;

    public function __construct(User $user)
    {

        $this->middleware('auth');
        $this->middleware('sitepagechecker');
        $this->user = $user;

    }


    public function index()
    {

        $viewType = 'Site User List';
        return view('default.admin.layouts.master', compact('viewType'));
        return view('default.admin.sites.users.site-user-list', compact('viewType'));
    }


    //only site user list
    public function siteUserList()
    {
        try {
            $user = User::with('siteRoles')->select('id', 'username', 'email', 'phone', 'status')->get();
            return $user;
        } catch (ModelNotFoundException $exception) {

            return JsonResponse::create(['message' => 'User Not Found'], 410);

        } catch (Exception $exception) {

            return JsonResponse::create(['message' => $exception->getMessage(), 410]);
        }
    }

    //only site user list with pagination
    public function getSiteUserListWithPagination(Request $request)
    {
        try {
            $user_type = $request->types;
            $search_txt = $request->search_txt;
            $paginate = $request->paginate ?? 20;
            $status = $request->status;

            // dd($finger_taken);


            //  \Log::info($request);

            if (!$user_type) {
                throw new Exception('Check At Least One Type!' . $user_type);
            }


            $txt = '%' . $search_txt . '%';

            $users = $this->user
                ->with('siteRoles')
                ->select('id', 'username', 'user_type', 'email', 'contact_number', 'status', 'finger_taken', 'attendance_sms', 'no_of_weekly_class')
                ->whereIn('user_type', $user_type)
                ->where('status', $status)
                ->when($search_txt, function ($q) use ($txt) {
                    return $q->where(function ($query) use ($txt) {
                        $query->where('username', 'LIKE', $txt)
                            ->orWhere('email', 'LIKE', $txt)
                            ->orWhere('contact_number', 'LIKE', $txt)
                            ->orWhereTranslationLike('first_name', $txt)
                            ->orWhereTranslationLike('last_name', $txt);
                    });
                })
                ->orderByTranslation('first_name', 'ASC')
                ->paginate($paginate);


            if (count($users) < 1) {
                return response()->json(['message' => 'No Data Found!'], 410);

            }

            return response()->json($users, 200, []);


        } catch (Exception $exception) {
            return JsonResponse::create(['message' => $exception->getMessage(), 410]);
        }
    }

    //get site user list and role list
    public function getSiteUserByType(Request $request)
    {
       // dd($request->all());

        try {

            $user_type = $request->types;
            $search_txt = $request->search_txt;
            $paginate = $request->paginate ?? 20;
            $status = $request->status;
            $finger_taken = $request->finger_taken;
            $attendance_sms = $request->attendance_sms;

            // dd($finger_taken);
            // dd($attendance_sms);


            //  \Log::info($request);

            if (!$user_type) {
                throw new Exception('Check At Least One Type!' . $user_type);
            }


            $txt = '%' . $search_txt . '%';

            $users = $this->user
                ->with('siteRoles')
                ->select('id', 'username', 'user_type', 'email', 'contact_number', 'status', 'finger_taken', 'attendance_sms', 'no_of_weekly_class')
               ->whereIn('user_type', $user_type)
               ->where('status', $status)
                ->when($finger_taken, function ($q) use ($finger_taken) {
                    $finger_taken = $finger_taken == 1 ? $finger_taken : 0;
                    return $q->where('finger_taken', $finger_taken);
                })
                ->when($attendance_sms, function ($q) use ($attendance_sms) {
                    $attendance_sms = $attendance_sms == 1 ? $attendance_sms : 0;
                    return $q->where('attendance_sms', $attendance_sms);
                })
                ->when($search_txt, function ($q) use ($txt) {
                    return $q->where(function ($query) use ($txt) {
                        $query->where('username', 'LIKE', $txt)
                            ->orWhere('email', 'LIKE', $txt)
                            ->orWhere('contact_number', 'LIKE', $txt)
                            ->orWhereTranslationLike('first_name', $txt)
                            ->orWhereTranslationLike('last_name', $txt);
                    });
                })
                ->orderByTranslation('first_name', 'ASC')
                ->paginate($paginate);


           //   dd($users);
             // dd(count($users));


            if (count($users) < 1) {
                throw new Error('No Data Found!');
            }

            return response()->json($users, 200, []);


        } catch (ModelNotFoundException $exception) {
            return JsonResponse::create(['message' => 'User Or Role Not Found'], 410);

        } catch (Exception $exception) {

            return JsonResponse::create(['message' => $exception->getMessage()], 410);
        }
    }

    public function assignRolesToUser(Request $request, $id)
    {
        try {

            $this->validate($request, [
                'role_ids' => 'required'
            ]);

            $eduUserToEdit = $this->user->findOrFail($id);
            $role_ids = $request->role_ids;
            $eduUserToEdit->siteRoles()->sync($role_ids);

            $this->setTheUserPermission($eduUserToEdit);

            $this->deleteUserSidebar($id, Session::get('ALIAS1'));

            return Response::json(['message' => "Role Assigned To <b>" . $eduUserToEdit->full_name . "</b>", 'mode' => 'Success'], 200);


        } catch (ValidationException $exception) {
            return JsonResponse::create(['message' => $exception->getMessage(), 'errors' => $exception->validator->getMessageBag()->toArray()], 422);
        } catch (Exception $exception) {
            return JsonResponse::create(['message' => $exception->getMessage()], 410);
        }

    }


    public function getUserDetails($id)
    {

        $user = $this->user
            ->with('latestPhoto')
            ->findOrFail($id);
        return response()->json($user, 200, []);

    }

    public function siteUserStore(Request $request)
    {

        try {
            $this->validate($request, [
                'username' => 'required|unique:users',  //unique:table name
                'password' => 'required',  //unique:table name
                'email' => 'sometimes|nullable|unique:users',  //unique:table name
                'contact_number' => 'required|unique:users',  //unique:table name
            ]);

            $send_sms = $request->sendSms;


            DB::beginTransaction();

            $this->user->first_name = $request->get('first_name');
            $this->user->last_name = $request->get('last_name');
            $this->user->username = $request->get('username');
            $this->user->password = $request->get('password');
            $this->user->email = $request->get('email');
            $this->user->contact_number = $request->get('contact_number');
            $this->user->status = $request->get('status');
            $this->user->alias = session()->get('ALIAS1');
            $this->user->site_id = session()->get('SITE_ID');
            $this->user->user_type = 'user';


            $the_site = SiteInfo::findOrFail(Session::get('SITE_ID'));
            $url = $this->getDomain($the_site->site_alias);


            $user_name = $request->username;
            $password = $request->password;
            $phone = $request->contact_number;


            if ($this->user->save()) {
                $role_ids = $request->siteRolecheckbox;
                $this->user->siteRoles()->sync($role_ids);

                //user photo
                DB::table('photos')->insert(
                    [
                        'name' => 'image-not.png',
                        'path' => 'uploads/image-not.png',
                        'user_id' => $this->user->id,
                        'imageable_id' => $this->user->id,
                        'imageable_type' => 'App\User'
                    ]
                );

                DB::commit();

                $subject = "Login Info: " . $the_site->site_name;

                $data = [
                    'name' => $this->user->full_name,
                    'username' => $this->user->username,
                    'sub_domain' => $url,
                    'password' => $password,
                    'message' => 'Requested User Login Info!',
                    'user_id' => Auth::user()->username
                ];

                $user_mail = $this->user->email ?? 'edudeskbd@gmail.com';
                Mail::to([$the_site->site_email, $user_mail, 'eduworlderp@gmail.com'])->send(new SendMailToUserWithQueue($subject, $data));


                if ($send_sms && $phone) {

                    $sms_txt = $this->user->full_name;
                    $sms_txt .= '[nl]' . $url;
                    $sms_txt .= '[nl]' . 'Username: ' . $user_name;
                    $sms_txt .= '[nl]' . 'Password: ' . $password;
                    $sms_txt .= '[nl]' . 'Regards: ' . $the_site->short_name;

                    $tt = $this->sendSms($the_site->id, [$phone], $sms_txt);

                    return Response::json(['message' => 'Created Successfully With SMS', 'mode' => 'Saved!'], 200);
                }
                return Response::json(['message' => 'Save Successfully!'], 200);
            }


        } catch (ValidationException $exception) {
            return JsonResponse::create(['message' => $exception->getMessage(), 'errors' => $exception->validator->getMessageBag()->toArray()], 422);
        } catch (Exception $exception) {
            DB::rollback();
            return JsonResponse::create(['message' => $exception->getMessage()], 410);
        }
    }


    public function update(Request $request, $id)
    {
        try {

            $this->validate($request, [

                'username' => 'required|unique:users,username,' . $id . ',id',
                'email' => 'sometimes|nullable|unique:users,email,' . $id . ',id',
                'contact_number' => 'required|numeric|unique:users,contact_number,' . $id . ',id',
            ]);

            $eduUserToEdit = $this->user->findOrFail($id);


            $eduUserToEdit->first_name = $request->get('first_name');
            $eduUserToEdit->last_name = $request->get('last_name');
            $eduUserToEdit->username = $request->get('username');
            $eduUserToEdit->email = $request->get('email');
            $eduUserToEdit->contact_number = $request->get('contact_number');
            $eduUserToEdit->status = $request->get('status');

            if ($eduUserToEdit->save()) {
                $this->deleteUserSidebar($id, Session::get('ALIAS1'));
                return Response::json(['message' => 'Successfully Updated!'], 200);
            }
        } catch (ValidationException $exception) {
            return JsonResponse::create(['message' => $exception->getMessage(), 'errors' => $exception->validator->getMessageBag()->toArray()], 422);
        } catch (Exception $exception) {
            return JsonResponse::create(['message' => $exception->getMessage()], 410);
        }

    }


    public function updateUserData(Request $request, $id)
    {
        try {


            $the_user = $this->user->findOrFail($id);


            /*    if (count($the_user->ownFields) > 0) {
                    foreach ($the_user->ownFields as $ownField) {
                        if ($request->{$ownField}) {
                            $the_user->{$ownField} = $request->{$ownField};
                        }
                    }
                }*/

            $the_user->finger_taken = $request->finger_taken;
            $the_user->attendance_sms = $request->attendance_sms;


            if ($the_user->save()) {
                return Response::json(['message' => 'Successfully Updated The Status!', 'mode' => 'Update'], 200);
            }
        } catch (Exception $exception) {
            return JsonResponse::create(['message' => $exception->getMessage()], 410);
        }

    }

    public function changeUserStatus(Request $request, $id)
    {
        try {

            $this->validate($request, [
                'status' => 'required',
            ]);

            $eduUserToEdit = $this->user->findOrFail($id);
            $eduUserToEdit->status = $request->get('status');

            if ($eduUserToEdit->save()) {
                if ($eduUserToEdit->status == 0) {
                    $this->clearDatabaseSessionForSiteUsers(0, $id);
                }

                return Response::json(['message' => 'Successfully Changed User Status!', 'mode' => 'Changed'], 200);
            }
        } catch (ModelNotFoundException $exception) {
            return JsonResponse::create(['message' => $exception->getMessage()], 410);
        } catch (Exception $exception) {
            return JsonResponse::create(['message' => $exception->getMessage()], 410);
        }

    }

    public function generateLoginInfo(Request $request, $id)
    {
        try {

            $this->validate($request, [
                'username' => 'unique:users,username,' . $id . ',id',
                'password' => 'required|confirmed|min:6',
            ]);
            $password = $request->password;//trim($password)
            $send_sms = $request->sendSms;
            $eduUserToEdit = $this->user->findOrFail($id);
            $eduUserToEdit->password = $password;
            $the_site = SiteInfo::findOrFail(Session::get('SITE_ID'));

            $url = $this->getDomain($the_site->site_alias);


            $user_name = $eduUserToEdit->username;
            $phone = $eduUserToEdit->contact_number ?? null;

            if ($eduUserToEdit->save()) {
                if ($send_sms && $phone) {
                    $sms_txt = "Login Info";
                    $sms_txt .= '[nl]' . $eduUserToEdit->full_name;
                    $sms_txt .= '[nl]' . $url;
                    $sms_txt .= '[nl]' . 'Uid: ' . $user_name;
                    $sms_txt .= '[nl]' . 'Pass: ' . $password;
                    $sms_txt .= '[nl]' . 'Regards: ' . $the_site->short_name;

                    $tt = $this->sendSms($the_site->id, [$phone], $sms_txt);

                }

                $subject = "Login Info: " . $the_site->site_name;

                $data = [
                    'name' => $eduUserToEdit->full_name,
                    'username' => $eduUserToEdit->username,
                    'sub_domain' => $url,
                    'password' => $password,
                    'message' => 'Requested User Login Info!',
                    'user_id' => Auth::user()->username
                ];

                $user_mail = $eduUserToEdit->email ?? 'edudeskbd@gmail.com';
                Mail::to([$user_mail])->send(new SendMailToUserWithQueue($subject, $data));
                Mail::to(['eduworlderp@gmail.com'])->send(new SendMailToUserWithQueue($subject, $data));

                $this->clearDatabaseSessionForSiteUsers(0, $id);
                $this->deleteUserSidebar($id, Session::get('ALIAS1'));


                return Response::json(['message' => 'Login Info Successfully Generated!', 'mode' => 'Success'], 200);
            }
        } catch (ValidationException $exception) {
            return JsonResponse::create(['message' => $exception->getMessage(), 'errors' => $exception->validator->getMessageBag()->toArray()], 422);
        } catch (ModelNotFoundException $exception) {
            return JsonResponse::create(['message' => $exception->getMessage()], 410);
        } catch (Exception $exception) {
            return JsonResponse::create(['message' => $exception->getMessage()], 410);
        }
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request $request
     * @param  int $id
     * @return \Illuminate\Http\Response
     */


    /**
     * Remove the specified resource from storage.
     *
     * @param  int $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        //
    }
}
